﻿/*
 * Copyright (c) 2013, M@T
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the <organization> nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL M@T BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System;
using System.Drawing;
using System.IO;
using System.Text;
using System.Windows.Forms;

namespace LibertyTicket_Distro_ROM_Editor
{
    public partial class frm_Main : Form
    {

        private uint arm9Offset; // = 0x04000;
        internal uint arm7Offset; // = 0x15C00;
        private uint dataOffset; // = 0x3CE00;
        private const uint PgfOffset = 0x4;
        private const uint TextOffset = 0xD4;
        private const uint DateOffset = 0x1684;
        private const uint JpOffset = 0x10DF;
        private const uint KrOffset = 0x13AF;
        private const uint GiftSize = 0x2D0;
        internal const uint BgOffset = 0x26398;
        private const uint ColorOffset = 0x12FC;

        internal static MemoryStream ms = new MemoryStream(0);
        private uint currentChar;
        private bool validROM;
        private bool canUpdateCardID = true;
        private int previousLangID = 1;
        private string romFilePath = "";

        private static string[] langIDs = { "en", "fr", "it", "de", "es", "jp", "kr" };

        public frm_Main()
        {
            InitializeComponent();
        }


        private void openRom(string romFile)
        {
            using (BinaryReader br = new BinaryReader(File.OpenRead(romFile)))
            {
                br.BaseStream.Position = 0;

                if (Encoding.UTF8.GetString(br.ReadBytes(18)) == "POKWBLIBERTYY8KP01")
                {
                    int fileLength = (int)br.BaseStream.Length;

                    if (fileLength > 0x800000 && MessageBox.Show("Le fichier spécifié a une taille supérieure à 8 Mio, qui est la taille normale de la ROM ; " +
                                                                 "ce n’est donc sans doute pas la ROM de distribution du Ticket Liberté.\r\n" +
                                                                 "Si la taille de ce fichier est trop grande, son chargement pourrait provoquer une consommation de mémoire importante.\r\n\r\n" +
                                                                 "Ouvrir ce fichier malgré tout ?", "Attention : fichier trop volumineux.",
                                                                 MessageBoxButtons.YesNo, MessageBoxIcon.Warning, MessageBoxDefaultButton.Button2) != DialogResult.Yes)
                    {
                        resetDisplay();
                        return;
                    }

                    validROM = true;
                    b_SaveROM.Enabled = true;
                    b_OpenPGF.Enabled = true;
                    b_SavePGF.Enabled = true;
                    cb_Lang.Enabled = true;
                    tb_Description.Enabled = true;
                    chk_DistribInf.Enabled = true;
                    lbl_CardID.Enabled = true;
                    nud_CardID.Enabled = true;
                    b_Cancel.Enabled = true;
                    tb_Title.Enabled = true;
                    b_EditHeader.Enabled = true;
                    b_ChooseColor.Enabled = true;
                    lbl_HtmlColor.Enabled = true;

                    romFilePath = romFile;
                    saveFileDialog1.InitialDirectory = Path.GetDirectoryName(romFilePath);

                    br.BaseStream.Position = 0;
                    ms.Position = 0;

                    byte[] buffer = new byte[fileLength];
                    br.Read(buffer, 0, fileLength);
                    ms.Write(buffer, 0, fileLength);

                    br.Close();

                    dataOffset = getDataOffset();
                    arm9Offset = getArm9Offset();
                    arm7Offset = getArm7Offset();

                    b_EditBG.Enabled = getArm7Size() == 0x3E398; // On n’active ce bouton que si la ROM supporte les images de fond

                    refreshData();
                    refreshColor();
                }
                else
                {
                    resetDisplay();
                    br.Close();
                    MessageBox.Show("Ce n’est pas la ROM de distribution du Ticket Liberté.",
                                    "Erreur : ROM invalide !", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
        }

        private void openPgf(string pgfFile)
        {
            using (BinaryReader br = new BinaryReader(File.OpenRead(pgfFile)))
            {

                if (br.BaseStream.Length == 204)
                {
                    ms.Position = GiftSize * cb_Lang.SelectedIndex + dataOffset + PgfOffset;
                    br.BaseStream.Position = 0;

                    ms.Write(br.ReadBytes(204), 0, 204);

                    br.Close();

                    saveText(cb_Lang.SelectedIndex);
                    refreshData();
                }
                else
                {
                    MessageBox.Show("Ce fichier PGF ne fait pas la bonne taille.\r\n\r\n (" + br.BaseStream.Length.ToString() + " octets au lieu de 204)",
                                    "Erreur : Fichier PGF invalide !", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
        }

        private void openRomInit()
        {
            openFileDialog1.Title = "Choisir la ROM de distribution à modifier";
            openFileDialog1.Filter = "ROMs NDS|*.nds";
            openFileDialog1.FileName = romFilePath == "" ? "" : Path.GetFileName(romFilePath);

            if (openFileDialog1.ShowDialog() == DialogResult.OK)
                openRom(openFileDialog1.FileName);
        }

        private void saveRomInit()
        {
            saveAllText(cb_Lang.SelectedIndex);
            ms.Position = dataOffset + DateOffset;

            if (chk_DistribInf.Checked)
            { // temps de distribution infini : du 01/01/0000 au 31/12/65535  :o)
                for (int i = 0; i < 8; i++)
                {
                    ms.WriteByte(0x00);
                    ms.WriteByte(0x00);
                    ms.WriteByte(0x01);
                    ms.WriteByte(0x01);
                    ms.WriteByte(0xFF);
                    ms.WriteByte(0xFF);
                    ms.WriteByte(0x0C);
                    ms.WriteByte(0x1F);
                }
            }
            else
            { // valeurs par défaut : du 25/02/2011 au 29/04/2011 (c’est comme ça dans la ROM, mais ça ne correspond pas aux dates annoncées)
                for (int i = 0; i < 8; i++)
                {
                    ms.WriteByte(0xDB);
                    ms.WriteByte(0x07);
                    ms.WriteByte(0x02);
                    ms.WriteByte(0x19);
                    ms.WriteByte(0xDB);
                    ms.WriteByte(0x07);
                    ms.WriteByte(0x04);
                    ms.WriteByte(0x1D);
                }
            }

            for (int i = 0; i < 7; i++)
            {
                ms.Position = dataOffset + 0xD2 + i * GiftSize;
                ms.WriteByte(0xF0);
            }

            ms.Position = dataOffset;
            ms.WriteByte(0x07);

            ms.Position = dataOffset + JpOffset;
            ms.WriteByte(0x01);

            ms.Position = dataOffset + KrOffset;
            ms.WriteByte(0x08);

            saveFileDialog1.Title = "Enregistrer la ROM de distribution modifiée";
            saveFileDialog1.Filter = "ROMs NDS|*.nds";
            saveFileDialog1.FileName = romFilePath == "" ? "" : Path.GetFileName(romFilePath);

            if (saveFileDialog1.ShowDialog() == DialogResult.OK)
            {
                string fileName = saveFileDialog1.FileName;
                if (File.Exists(fileName))
                    File.Delete(fileName);
                File.WriteAllBytes(fileName, ms.ToArray());
            }
        }

        private void refreshData()
        {
            ms.Position = GiftSize * cb_Lang.SelectedIndex + dataOffset + PgfOffset + 0x60;
            tb_Title.Text = "";

            for (int i = 0; i < 0x24; i++)
            { // obtient le titre du PCD de la langue courante
                currentChar = (uint)(ms.ReadByte() | ms.ReadByte() * 0x100);

                if (currentChar == 0xFFFF)
                    break;

                tb_Title.Text += (char)(currentChar);
            }

            ms.Position = GiftSize * cb_Lang.SelectedIndex + dataOffset + TextOffset;
            tb_Description.Text = "";

            for (int i = 0; i < 0xFB; i++)
            {
                currentChar = (uint)(ms.ReadByte() | ms.ReadByte() * 0x100);

                if (currentChar == 0xFFFF)
                    break;

                else if (currentChar == 0xFFFE)
                    tb_Description.Text += "\r\n";

                else
                    tb_Description.Text += (char)(currentChar);
            }

            ms.Position = GiftSize * cb_Lang.SelectedIndex + dataOffset + PgfOffset + 0xB0;

            canUpdateCardID = false;
            nud_CardID.Value = ms.ReadByte() | ms.ReadByte() * 0x100;
            canUpdateCardID = true;
        }

        private void refreshColor()
        {
            ms.Position = arm9Offset + ColorOffset;
            Color newColor = abgr1555ToArgb8888((ushort)(ms.ReadByte() | ms.ReadByte() * 0x100));

            colorDialog1.Color = newColor;

            uint htmlColor = (uint)((newColor.R << 16) | (newColor.G << 8) | newColor.B);

            if (newColor.GetBrightness() < 0.5)
                lbl_HtmlColor.BackColor = Color.White;
            else
                lbl_HtmlColor.BackColor = Color.Black;

            lbl_HtmlColor.ForeColor = newColor;
            lbl_HtmlColor.Text = "#" + htmlColor.ToString("X6");
        }

        private void saveGiftTitle(int langID)
        {
            ms.Position = GiftSize * langID + dataOffset + PgfOffset + 0x60; // titre du Cadeau Mystère

            for (int i = 0; i < 0x4A; i++) // efface tout le texte en le remplaçant par des 0xFF
                ms.WriteByte(0xFF);

            ms.Position = GiftSize * langID + dataOffset + PgfOffset + 0x60;

            for (int i = 0; i < tb_Title.Text.Length; i++)
            {
                ms.WriteByte((byte)(tb_Title.Text[i] & 0xFF));
                ms.WriteByte((byte)(tb_Title.Text[i] >> 8));
            }
        }

        private void saveAllText(int langID)
        {
            saveText(langID);
            saveGiftTitle(langID);
        }

        private void saveText(int langID)
        {
            ms.Position = GiftSize * langID + dataOffset + TextOffset;

            for (int i = 0; i < 0x1FA; i++) // efface tout le texte en le remplaçant par des 0xFF
                ms.WriteByte(0xFF);

            ms.Position = GiftSize * langID + dataOffset + TextOffset;

            for (int i = 0; i < tb_Description.Text.Length; i++)
            {
                if (tb_Description.Text[i] == '\r' && tb_Description.Text[i + 1] == '\n')
                { // retour à la ligne
                    ms.WriteByte(0xFE);
                    ms.WriteByte(0xFF);
                    i++; // on incrémente pour sauter le '\n'
                }
                else if (tb_Description.Text[i] == '\n')
                { // en cas de retours à la ligne style Unix (n’est pas censé arriver sous Windows mais on sait jamais...)
                    ms.WriteByte(0xFE);
                    ms.WriteByte(0xFF);
                }
                else
                {
                    ms.WriteByte((byte)(tb_Description.Text[i] & 0xFF));
                    ms.WriteByte((byte)(tb_Description.Text[i] >> 8));
                }
            }
        }

        private uint getArm9Offset()
        {
            BinaryReader br = new BinaryReader(ms);
            br.BaseStream.Position = 0x20;
            return br.ReadUInt32();
        }

        private uint getArm7Offset()
        {
            BinaryReader br = new BinaryReader(ms);
            br.BaseStream.Position = 0x30;
            return br.ReadUInt32();
        }

        private uint getDataOffset()
        {
            BinaryReader br = new BinaryReader(ms);
            br.BaseStream.Position = 0x48;
            br.BaseStream.Position = br.ReadUInt32();
            return br.ReadUInt32();
        }

        private uint getArm7Size()
        {
            BinaryReader br = new BinaryReader(ms);
            br.BaseStream.Position = 0x3C;
            return br.ReadUInt32();
        }

        private void resetDisplay()
        {
            validROM = false;
            b_SaveROM.Enabled = false;
            b_OpenPGF.Enabled = false;
            b_SavePGF.Enabled = false;
            cb_Lang.Enabled = false;
            tb_Description.Enabled = false;
            chk_DistribInf.Enabled = false;
            lbl_CardID.Enabled = false;
            nud_CardID.Enabled = false;
            b_Cancel.Enabled = false;
            tb_Title.Enabled = false;
            b_EditHeader.Enabled = false;
            b_EditBG.Enabled = false;
            b_ChooseColor.Enabled = false;
            lbl_HtmlColor.Enabled = false;
            tb_Title.Text = "";
            tb_Description.Text = "";
            lbl_CharCnt.Text = "";
            cb_Lang.SelectedIndex = 1;
            nud_CardID.Value = 0;
        }

        private ushort argb8888ToAbgr1555(Color rgb)
        {
            byte r5 = (byte)(rgb.R >> 3),
                 g5 = (byte)(rgb.G >> 3),
                 b5 = (byte)(rgb.B >> 3),
                 a1 = (byte)(rgb.A >> 7);

            return (ushort)(r5 | (g5 << 5) | (b5 << 10) | a1 << 15);
        }

        private Color abgr1555ToArgb8888(ushort abgr)
        {
            byte r5 = (byte)(abgr & 0x001F),
                 g5 = (byte)((abgr & 0x03E0) >> 5),
                 b5 = (byte)((abgr & 0x7C00) >> 10),
                 a1 = (byte)((abgr & 0x8000) >> 15);

            return Color.FromArgb(a1 * 255, (r5 << 3) | (r5 >> 2), (g5 << 3) | (g5 >> 2), (b5 << 3) | (b5 >> 2));
        }

        private void showFrm_Header()
        {
            using (frm_Header f = new frm_Header())
            {
                f.Owner = this;
                f.ShowDialog();
                f.Dispose();
            }
        }

        private void showFrm_Image()
        {
            using (frm_Image f = new frm_Image())
            {
                f.Owner = this;
                f.ShowDialog();
                f.Dispose();
            }
        }

        private void frm_Main_Load(object sender, EventArgs e)
        {
            cb_Lang.SelectedIndex = 1;

            string[] args = Environment.GetCommandLineArgs();
            if (args.Length > 1 && File.Exists(args[1]))
                openRom(args[1]);
        }

        private void frm_Main_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.Control)
                switch (e.KeyCode)
                {
                    case Keys.E:
                        if (validROM)
                        {
                            e.SuppressKeyPress = true;
                            BeginInvoke(new MethodInvoker(showFrm_Header));
                        }
                        break;

                    case Keys.O:
                        e.SuppressKeyPress = true;
                        BeginInvoke(new MethodInvoker(openRomInit));
                        break;

                    case Keys.S:
                        if (validROM)
                        {
                            e.SuppressKeyPress = true;
                            BeginInvoke(new MethodInvoker(saveRomInit));
                        }
                        break;
                }

        }

        private void frm_Main_DragDrop(object sender, DragEventArgs e)
        {
            string[] fileNames = (string[])e.Data.GetData(DataFormats.FileDrop, true);
            string ndsFile = null, pgfFile = null;

            foreach (string fileName in fileNames)
            {
                switch (Path.GetExtension(fileName).ToLowerInvariant())
                {
                    case ".pgf":
                        pgfFile = fileName;
                        break;

                    case ".nds":
                        ndsFile = fileName;
                        break;
                }
            }

            if (ndsFile != null)
                openRom(ndsFile);

            if (pgfFile != null && validROM)
                openPgf(pgfFile);
        }

        private void frm_Main_DragEnter(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
                e.Effect = DragDropEffects.Copy;
            else
                e.Effect = DragDropEffects.None;
        }

        private void cb_Lang_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (validROM)
            {
                saveAllText(previousLangID);
                previousLangID = cb_Lang.SelectedIndex;
                refreshData();
            }
        }

        private void b_OpenROM_Click(object sender, EventArgs e)
        {
            openRomInit();
        }

        private void b_OpenPGF_Click(object sender, EventArgs e)
        {
            openFileDialog1.Title = "Ouvrir un fichier de Cadeau Mystère 5G";
            openFileDialog1.Filter = "Fichiers Cadeau Mystère 5G|*.pgf";
            openFileDialog1.FileName = "";

            if (openFileDialog1.ShowDialog() == DialogResult.OK)
            {
                openPgf(openFileDialog1.FileName);
            }
        }

        private void b_SavePGF_Click(object sender, EventArgs e)
        {
            saveFileDialog1.Title = "Enregistrer le fichier du Cadeau Mystère";
            saveFileDialog1.Filter = "Fichiers Cadeau Mystère 5G|*.pgf";
            saveFileDialog1.FileName = "gift_" + (int)nud_CardID.Value + "_" + langIDs[cb_Lang.SelectedIndex] + ".pgf";

            if (saveFileDialog1.ShowDialog() == DialogResult.OK)
            {
                saveGiftTitle(cb_Lang.SelectedIndex);

                ms.Position = GiftSize * cb_Lang.SelectedIndex + dataOffset + PgfOffset;

                byte[] buffer = new byte[204];
                ms.Read(buffer, 0, 204);

                string fileName = saveFileDialog1.FileName;
                if (File.Exists(fileName))
                    File.Delete(fileName);
                File.WriteAllBytes(fileName, buffer);
            }
        }

        private void b_SaveROM_Click(object sender, EventArgs e)
        {
            saveRomInit();
        }

        private void nud_CardID_ValueChanged(object sender, EventArgs e)
        {
            if (canUpdateCardID)
            {
                ms.Position = GiftSize * cb_Lang.SelectedIndex + dataOffset + PgfOffset + 0xB0;
                ms.WriteByte((byte)((int)nud_CardID.Value & 0xFF));
                ms.WriteByte((byte)((int)nud_CardID.Value >> 8));
            }

        }

        private void tb_Description_TextChanged(object sender, EventArgs e)
        {
            int crLfCnt = (tb_Description.Text.Length - tb_Description.Text.Replace("\r\n", "").Length) / 2;

            tb_Description.MaxLength = 252 + crLfCnt;

            lbl_CharCnt.Text = (tb_Description.Text.Length - crLfCnt).ToString() + "/252";
        }

        private void b_Cancel_Click(object sender, EventArgs e)
        {
            if (MessageBox.Show("Êtes-vous sûr de vouloir annuler toutes les modifications ?", "Voulez-vous vraiment annuler ?", MessageBoxButtons.YesNo, MessageBoxIcon.Question, MessageBoxDefaultButton.Button2) == DialogResult.Yes)
            {
                if (File.Exists(romFilePath))
                {
                    using (BinaryReader br = new BinaryReader(File.OpenRead(romFilePath)))
                    {
                        br.BaseStream.Position = 0;

                        if (Encoding.UTF8.GetString(br.ReadBytes(18)) == "POKWBLIBERTYY8KP01")
                        {
                            br.BaseStream.Position = 0;
                            ms.Position = 0;

                            int count = (int)br.BaseStream.Length;

                            byte[] buffer = new byte[count];
                            br.Read(buffer, 0, count);
                            ms.Write(buffer, 0, count);

                            br.Close();

                            dataOffset = getDataOffset();
                            arm9Offset = getArm9Offset();
                            arm7Offset = getArm7Offset();

                            b_EditBG.Enabled = getArm7Size() == 0x3E398; // On n’active ce bouton que si la ROM supporte les images de fond

                            refreshData();
                            refreshColor();
                        }
                        else
                        {
                            br.Close();
                            MessageBox.Show("La ROM n’est plus valide.\r\nAnnulation de la procédure d’annulation.  :j",
                                            "Attention : La ROM n’est plus valide.", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                        }
                    }
                }
                else
                {
                    MessageBox.Show("Le fichier n’existe plus.\r\nAnnulation de la procédure d’annulation.  :j", "Attention : Le fichier n’existe plus.", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                }
            }
        }

        private void b_EditHeader_Click(object sender, EventArgs e)
        {
            showFrm_Header();
        }

        private void b_EditBG_Click(object sender, EventArgs e)
        {
            showFrm_Image();
        }

        private void showColorDialog()
        {
            if (colorDialog1.ShowDialog() == DialogResult.OK)
            {
                Color color = colorDialog1.Color;
                ushort abgrColor = argb8888ToAbgr1555(color);

                ms.Position = arm9Offset + ColorOffset;
                ms.WriteByte((byte)(abgrColor & 0xFF));
                ms.WriteByte((byte)(abgrColor >> 8));

                refreshColor();
            }
        }

        private void b_ChooseColor_Click(object sender, EventArgs e)
        {
            showColorDialog();
        }

        private void lbl_HtmlColor_DoubleClick(object sender, EventArgs e)
        {
            showColorDialog();
        }
    }
}
